' SAP Tool Backlog Data Extraction Script
' Author: Dario Pascoal
'
' Description: This script automates the extraction of tool backlog data from SAP by combining
' information from two different transactions: IH10 (Stock Overview) and ZSTK (Custom Stock Report).
' The script connects to SAP, executes both transactions with specified filters, exports the results
' to temporary files, then processes and merges the data to create a comprehensive tool inventory
' report with serial numbers, equipment details, plant/storage location information, bin locations,
' and change dates.
'
' The business purpose is to provide DHL operations teams with current tool inventory status
' across multiple plants and storage locations, enabling better resource planning and tool
' tracking. The merged dataset combines detailed equipment information from IH10 with
' storage location specifics from ZSTK to give a complete picture of tool availability.
'
' Prerequisites: Active SAP GUI session with access to IH10 and ZSTK transactions
' Parameters:
'   - Argument 1: Plant code(s) (optional) - Single plant or comma-separated list (e.g. "NL10" or "NL10,NL05")
'   - Argument 2: Storage location(s) (optional) - Single SLOC or comma-separated list (e.g. "D008" or "D008,D009")
'   - Argument 3: Data output path (optional) - Where to save the merged results file
'   - Argument 4: Log file path (optional) - Path for detailed logging output
' Returns: Merged dataset file containing comprehensive tool backlog information
'
On Error Resume Next

' Configure SAP GUI automation environment for script execution
' This enables programmatic control of the SAP interface and disables security warnings
Set WSHShell = CreateObject("WScript.Shell")

' Disable all SAP GUI scripting security warnings in Windows registry
' These settings prevent popup dialogs that would interrupt the automated data extraction process
' Each registry entry controls a different aspect of SAP GUI security warnings
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\FileWriteAccess", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\FileDirWriteAccess", 0, "REG_DWORD"

' Initialize command line parameters for flexible script execution
' These parameters allow the script to be customized for different plants and storage locations
Dim plant, storageLocation, dataOutputPath, tmpOutputPath
plant = ""
storageLocation = ""

' Establish file system access and determine application directory structure
' This section sets up the necessary file paths for temporary data storage and output
Set fso = CreateObject("Scripting.FileSystemObject")
Dim appFolder, scriptPath, programFolder
scriptPath = WScript.ScriptFullName
appFolder = fso.GetParentFolderName(scriptPath)
programFolder = fso.GetParentFolderName(fso.GetParentFolderName(appFolder))

' Output debug information to help diagnose file location issues
' This is essential for troubleshooting when files are not saved where expected
WScript.Echo "DEBUG:scriptPath=" & scriptPath
WScript.Echo "DEBUG:appFolder=" & appFolder
WScript.Echo "DEBUG:programFolder=" & programFolder

' Create temporary folder for SAP export files if it doesn't exist
' The tmp folder stores intermediate files from SAP exports before processing
tmpOutputPath = programFolder & "\tmp"
If Not fso.FolderExists(tmpOutputPath) Then
    On Error Resume Next
    fso.CreateFolder(tmpOutputPath)
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not create tmp folder: " & Err.Description
        ' Fallback to Windows temp folder if program directory is not writable
        tmpOutputPath = fso.GetSpecialFolder(2) ' Windows temp folder
        WScript.Echo "DEBUG:Falling back to: " & tmpOutputPath
    End If
    On Error GoTo 0
End If

WScript.Echo "DEBUG:Using tmpOutputPath=" & tmpOutputPath

' Process command line arguments to configure script execution parameters
' This flexible parameter system allows the script to be customized for different scenarios
If WScript.Arguments.Count > 0 Then
    plant = WScript.Arguments.Item(0)
End If

If WScript.Arguments.Count > 1 Then
    storageLocation = WScript.Arguments.Item(1)
End If

If WScript.Arguments.Count > 2 Then
    dataOutputPath = WScript.Arguments.Item(2)
Else
    ' Default output location in tmp folder if not specified
    dataOutputPath = tmpOutputPath & "\sap_backlog_data.txt"
End If

' Initialize logging system for debugging and audit trail purposes
' Variables to handle comprehensive logging throughout the script execution
Dim logFile, logFilePath

' Set up optional detailed logging if log file path is provided as fourth argument
' This enables comprehensive tracking of script execution for troubleshooting and auditing
If WScript.Arguments.Count > 3 Then
    logFilePath = WScript.Arguments.Item(3)
    ' Attempt to open log file in append mode to preserve existing log entries
    On Error Resume Next
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Create comprehensive log entry header for this script execution
        LogToFile "========================================="
        LogToFile "Backlog Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Plant: " & plant
        LogToFile "Storage Location: " & storageLocation
        LogToFile "Data Output Path: " & dataOutputPath
        LogToFile "Temp Output Path: " & tmpOutputPath
        LogToFile "========================================="
    End If
End If

' Helper function to write messages to log file only
' This supports detailed debugging and audit trails without cluttering console output
' Parameters: message - The text to write to the log file with timestamp and script identification
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Write formatted log entry with timestamp and script identification
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Helper function to output messages to both console and log file
' This ensures important information is visible immediately and permanently recorded
' Parameters: message - The text to display to user and preserve in log
Sub LogMessage(message)
    WScript.Echo message    ' Provide immediate feedback to user
    LogToFile message       ' Preserve detailed record in log file
End Sub

' Initialize output file for the final merged dataset
' This file will contain the comprehensive tool backlog information
Dim outputFile, dataCollected
Set outputFile = fso.CreateTextFile(dataOutputPath, True)
dataCollected = False

' Define file paths for temporary SAP export files
' These intermediate files store raw data from each SAP transaction
Dim ih10FilePath, zstkFilePath
ih10FilePath = tmpOutputPath & "\IH10_data.txt"
zstkFilePath = tmpOutputPath & "\ZSTK_data.txt"

On Error Resume Next

' Establish connection to existing SAP GUI session
' This script requires an active SAP session to be already running
' The connection hierarchy follows: Application -> Connection -> Session
If Not IsObject(application) Then
   Set SapGuiAuto = GetObject("SAPGUI")
   Set application = SapGuiAuto.GetScriptingEngine
End If
If Not IsObject(connection) Then
   Set connection = application.Children(0)
End If
If Not IsObject(session) Then
   Set session = connection.Children(0)
End If
' Enable event handling for SAP GUI automation
If IsObject(WScript) Then
   WScript.ConnectObject session, "on"
   WScript.ConnectObject application, "on"
End If

LogMessage "Starting backlog data collection..."

' Ensure SAP window is maximized for consistent element access
session.findById("wnd[0]").maximize

' ============================================================
' PART 1: IH10 TRANSACTION EXECUTION
' IH10 provides stock overview information including equipment details,
' serial numbers, and plant/storage location data with change dates
' ============================================================

LogMessage "Opening IH10 transaction..."
session.findById("wnd[0]/tbar[0]/okcd").text = "/nIH10"
session.findById("wnd[0]").sendVKey 0

' Configure IH10 filters based on provided parameters
' The plant filter supports both single plants and comma-separated lists for multi-plant operations
If plant <> "" Then
    LogMessage "Setting plant: " & plant
    
    ' Handle multiple plants using comma-separated values
    ' This allows operations teams to extract data from multiple facilities simultaneously
    If InStr(plant, ",") > 0 Then
        LogMessage "Multiple plants detected: " & plant
        plantArray = Split(plant, ",")
        
        ' Open multi-selection dialog for plant values
        session.findById("wnd[0]/usr/btn%_WERK_%_APP_%-VALU_PUSH").press
        
        ' Add each plant from the comma-separated list to the selection
        For i = 0 To UBound(plantArray)
            currPlant = Trim(plantArray(i))
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currPlant
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currPlant)
        Next
        
        ' Confirm the multi-selection
        session.findById("wnd[1]/tbar[0]/btn[8]").press
    Else
        ' Single plant - set directly in the main field
        session.findById("wnd[0]/usr/ctxtWERK-LOW").text = plant
        session.findById("wnd[0]/usr/ctxtWERK-LOW").setFocus
        session.findById("wnd[0]/usr/ctxtWERK-LOW").caretPosition = Len(plant)
    End If
End If

' Configure storage location filter with support for multiple locations
' Storage locations define specific areas within plants where tools are stored
If storageLocation <> "" Then
    LogMessage "Setting storage location: " & storageLocation
    
    ' Handle multiple storage locations using comma-separated values
    ' This enables extraction from multiple storage areas within the same operation
    If InStr(storageLocation, ",") > 0 Then
        LogMessage "Multiple storage locations detected: " & storageLocation
        slocArray = Split(storageLocation, ",")
        
        ' Open multi-selection dialog for storage location values
        session.findById("wnd[0]/usr/btn%_LAGER_%_APP_%-VALU_PUSH").press
        
        ' Add each storage location from the comma-separated list
        For i = 0 To UBound(slocArray)
            currSloc = Trim(slocArray(i))
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currSloc
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currSloc)
        Next
        
        ' Confirm the storage location selection
        session.findById("wnd[1]/tbar[0]/btn[8]").press
    Else
        ' Single storage location - set directly in the main field
        session.findById("wnd[0]/usr/ctxtLAGER-LOW").text = storageLocation
        session.findById("wnd[0]/usr/ctxtLAGER-LOW").setFocus
        session.findById("wnd[0]/usr/ctxtLAGER-LOW").caretPosition = Len(storageLocation)
    End If
End If

' Execute IH10 transaction with predefined variant for consistent output format
' The variant /BCKLGFRTVDR ensures we get the specific columns needed for backlog analysis
LogMessage "Executing IH10 with variant"
session.findById("wnd[0]/usr/ctxtVARIANT").text = "/BCKLGFRTVDR"
session.findById("wnd[0]/usr/ctxtVARIANT").setFocus
session.findById("wnd[0]/usr/ctxtVARIANT").caretPosition = 12
session.findById("wnd[0]/tbar[1]/btn[8]").press

' Export IH10 results to temporary file for processing
' This creates a tab-delimited text file containing all the stock overview data
LogMessage "Exporting IH10 data..."
On Error Resume Next
session.findById("wnd[0]/mbar/menu[0]/menu[10]/menu[2]").select

' Handle different SAP menu structures with fallback option
' Some SAP versions may have slightly different menu paths
If Err.Number <> 0 Then
    Err.Clear
    session.findById("wnd[0]/mbar/menu[0]/menu[3]/menu[1]").select
End If

' Configure export format as unconverted (tab-delimited text)
' This preserves all data formatting and ensures reliable parsing
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").select
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").setFocus
session.findById("wnd[1]/tbar[0]/btn[0]").press

' Specify export destination and filename for IH10 data
session.findById("wnd[1]/usr/ctxtDY_PATH").text = tmpOutputPath
session.findById("wnd[1]/usr/ctxtDY_FILENAME").text = "IH10_data.txt"
session.findById("wnd[1]/usr/ctxtDY_FILENAME").caretPosition = 13
session.findById("wnd[1]/tbar[0]/btn[11]").press

' Allow time for file export completion and return to main SAP screen
WScript.Sleep 2000
session.findById("wnd[0]").sendVKey 3
session.findById("wnd[0]").sendVKey 3

' ============================================================
' PART 2: ZSTK TRANSACTION EXECUTION
' ZSTK is a custom report that provides additional stock information
' including bin locations and profile data not available in IH10
' ============================================================

LogMessage "Opening ZSTK transaction..."
session.findById("wnd[0]/tbar[0]/okcd").text = "/nZSTK"
session.findById("wnd[0]").sendVKey 0

' Clear any pre-filled material number to ensure we get all materials
session.findById("wnd[0]/usr/ctxtSO_MATNR-LOW").text = ""

' Initialize variables for handling multiple plants and storage locations in ZSTK
' ZSTK requires different handling than IH10 for multi-value selections
Dim isMultiplePlantsZSTK, isMultipleSlocsZSTK
Dim plantArrayZSTK, slocArrayZSTK ' Use ZSTK-specific array names to avoid conflicts

isMultiplePlantsZSTK = False
isMultipleSlocsZSTK = False

' Configure plant filter for ZSTK transaction
' ZSTK uses a different field structure than IH10, requiring adapted handling
If plant <> "" Then
    LogMessage "Processing plant for ZSTK: " & plant
    If InStr(plant, ",") > 0 Then
        ' Multiple plants detected - handle the first one in main field, others in multi-select
        isMultiplePlantsZSTK = True
        plantArrayZSTK = Split(plant, ",")
        If UBound(plantArrayZSTK) >= 0 Then
            Dim firstPlantZSTK
            firstPlantZSTK = Trim(plantArrayZSTK(0))
            session.findById("wnd[0]/usr/ctxtSO_WERKS-LOW").text = firstPlantZSTK
            LogMessage "Set first plant '" & firstPlantZSTK & "' in ZSTK main field."
        Else
            LogMessage "Warning: Plant string '" & plant & "' resulted in an empty array for ZSTK."
            isMultiplePlantsZSTK = False ' Treat as no valid multiple plants
        End If
    Else
        ' Single plant - set directly in the main field
        session.findById("wnd[0]/usr/ctxtSO_WERKS-LOW").text = plant
        LogMessage "Set single plant '" & plant & "' in ZSTK main field."
    End If
Else
    LogMessage "No plant provided for ZSTK. ZSTK might require a plant."
    ' ZSTK may require plant specification - proceeding with execution to handle gracefully
End If

' Configure storage location filter for ZSTK transaction
' This provides the bin location data that complements IH10's equipment information
If storageLocation <> "" Then
    LogMessage "Processing storage location for ZSTK: " & storageLocation
    If InStr(storageLocation, ",") > 0 Then
        ' Multiple storage locations - handle first one in main field, others in multi-select
        isMultipleSlocsZSTK = True
        slocArrayZSTK = Split(storageLocation, ",")
        If UBound(slocArrayZSTK) >= 0 Then
            Dim firstSlocZSTK
            firstSlocZSTK = Trim(slocArrayZSTK(0))
            session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").text = firstSlocZSTK
            session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").setFocus ' Ensure field activation in SAP
            session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").caretPosition = Len(firstSlocZSTK)
            LogMessage "Set first SLOC '" & firstSlocZSTK & "' in ZSTK main field."
        Else
            LogMessage "Warning: Storage location string '" & storageLocation & "' resulted in an empty array for ZSTK."
            isMultipleSlocsZSTK = False ' Treat as no valid multiple storage locations
        End If
    Else
        ' Single storage location - set directly in the main field
        session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").text = storageLocation
        session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").setFocus
        session.findById("wnd[0]/usr/ctxtSO_LGORT-LOW").caretPosition = Len(storageLocation)
        LogMessage "Set single SLOC '" & storageLocation & "' in ZSTK main field."
    End If
Else
    LogMessage "No storage location provided for ZSTK. ZSTK might require a storage location."
    ' ZSTK may require storage location specification - proceeding to handle gracefully
End If

' Handle multi-selection for additional plants beyond the first one
' This step is performed after the main fields are populated to add remaining plant values
If isMultiplePlantsZSTK And UBound(plantArrayZSTK) > 0 Then ' More than one plant in the original list
    LogMessage "Opening multi-selection for remaining " & UBound(plantArrayZSTK) & " plants in ZSTK."
    session.findById("wnd[0]/usr/btn%_SO_WERKS_%_APP_%-VALU_PUSH").press
    
    Dim currentPlantZSTKLoop
    ' The first plant is already in the main field and auto-populates row 0 of the multi-select dialog
    ' Additional plants are added starting from row 1 of the dialog table
    For i = 1 To UBound(plantArrayZSTK) ' Start from second item in array (index 1)
        currentPlantZSTKLoop = Trim(plantArrayZSTK(i))
        If currentPlantZSTKLoop <> "" Then
            LogMessage "Adding plant '" & currentPlantZSTKLoop & "' to ZSTK multi-select dialog at table row " & i
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currentPlantZSTKLoop
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currentPlantZSTKLoop)
        End If
    Next
    
    LogMessage "Confirming ZSTK plant multi-selection."
    session.findById("wnd[1]/tbar[0]/btn[8]").press ' Confirm selection
ElseIf isMultiplePlantsZSTK And UBound(plantArrayZSTK) = 0 Then
    LogMessage "Only one plant detected for ZSTK (from '" & plant & "'), already set in main field. No multi-selection needed for plant."
End If


' Handle multi-selection for additional storage locations beyond the first one
' This process mirrors the plant multi-selection but for storage location fields
If isMultipleSlocsZSTK And UBound(slocArrayZSTK) > 0 Then ' More than one storage location in the original list
    LogMessage "Opening multi-selection for remaining " & UBound(slocArrayZSTK) & " SLOCs in ZSTK."
    session.findById("wnd[0]/usr/btn%_SO_LGORT_%_APP_%-VALU_PUSH").press
    
    Dim currentSlocZSTKLoop
    ' Similar to plants: first storage location is in main field and auto-populates row 0
    ' Additional storage locations are added starting from row 1 of the dialog table
    For i = 1 To UBound(slocArrayZSTK) ' Start from second item in array (index 1)
        currentSlocZSTKLoop = Trim(slocArrayZSTK(i))
        If currentSlocZSTKLoop <> "" Then
            LogMessage "Adding SLOC '" & currentSlocZSTKLoop & "' to ZSTK multi-select dialog at table row " & i
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").text = currentSlocZSTKLoop
            session.findById("wnd[1]/usr/tabsTAB_STRIP/tabpSIVA/ssubSCREEN_HEADER:SAPLALDB:3010/tblSAPLALDBSINGLE/ctxtRSCSEL_255-SLOW_I[1," & i & "]").caretPosition = Len(currentSlocZSTKLoop)
        End If
    Next
    
    ' Confirm storage location selection with error handling for different SAP versions
    LogMessage "Confirming ZSTK SLOC multi-selection."
    On Error Resume Next
    session.findById("wnd[1]/tbar[0]/btn[8]").press
    If Err.Number <> 0 Then
        LogMessage "Error confirming ZSTK SLOC selection with btn[8]: " & Err.Description
        Err.Clear
        LogMessage "Trying alternative confirm button btn[0] for ZSTK SLOC."
        session.findById("wnd[1]/tbar[0]/btn[0]").press
        If Err.Number <> 0 Then
            LogMessage "Failed alternative confirm button btn[0] for ZSTK SLOC: " & Err.Description
        End If
    End If
    On Error GoTo 0
ElseIf isMultipleSlocsZSTK And UBound(slocArrayZSTK) = 0 Then
    LogMessage "Only one SLOC detected for ZSTK (from '" & storageLocation & "'), already set in main field. No multi-selection needed for SLOC."
End If

' Execute ZSTK report with configured filters
LogMessage "Executing ZSTK search..."
session.findById("wnd[0]/tbar[1]/btn[8]").press

' Access layout configuration to select specific columns for export
' This ensures we get only the data we need for the backlog analysis
session.findById("wnd[0]/tbar[1]/btn[32]").press

' Configure column selection for ZSTK export
' These specific column selections ensure we get material, plant, storage location, bin, and profile data
' The multiple button presses are navigating through the column selection interface
On Error Resume Next
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").currentCellRow = 2
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").selectedRows = "2"
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").currentCellRow = 7
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/cntlCONTAINER2_LAYO/shellcont/shell").selectedRows = "7"
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/usr/tabsG_TS_ALV/tabpALV_M_R1/ssubSUB_DYN0510:SAPLSKBH:0620/btnAPP_FL_SING").press
session.findById("wnd[1]/tbar[0]/btn[0]").press

' Export ZSTK results to temporary file for processing
' This creates a complementary dataset with bin location and profile information
LogMessage "Exporting ZSTK data..."
session.findById("wnd[0]/mbar/menu[0]/menu[1]/menu[2]").select
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").select
session.findById("wnd[1]/usr/subSUBSCREEN_STEPLOOP:SAPLSPO5:0150/sub:SAPLSPO5:0150/radSPOPLI-SELFLAG[1,0]").setFocus
session.findById("wnd[1]/tbar[0]/btn[0]").press

' Specify export destination and filename for ZSTK data
session.findById("wnd[1]/usr/ctxtDY_PATH").text = tmpOutputPath
session.findById("wnd[1]/usr/ctxtDY_FILENAME").text = "ZSTK_data.txt"
session.findById("wnd[1]/usr/ctxtDY_FILENAME").caretPosition = 13
session.findById("wnd[1]/tbar[0]/btn[11]").press

' Allow time for file export completion and return to main SAP screen
WScript.Sleep 2000
session.findById("wnd[0]").sendVKey 3
session.findById("wnd[0]").sendVKey 3

' ============================================================
' PART 3: DATA PROCESSING AND MERGING
' Process the exported files from both transactions and merge
' them into a comprehensive tool backlog dataset
' ============================================================

LogMessage "Processing exported data..."
On Error Resume Next

' Verify both export files were created successfully before proceeding
If fso.FileExists(ih10FilePath) And fso.FileExists(zstkFilePath) Then
    LogMessage "Both export files found. Processing data..."
    
    ' Initialize file reading variables and data structures
    Dim ih10File, zstkFile
    Dim ih10Line, zstkLine, ih10Fields, zstkFields, i, j
    
    ' Create dictionary objects to efficiently store and cross-reference data
    ' Dictionaries provide fast lookup capabilities for merging datasets
    Dim ih10Records, zstkRecords, zstkData
    Set ih10Records = CreateObject("Scripting.Dictionary")
    Set zstkRecords = CreateObject("Scripting.Dictionary")
    
    ' Open both exported files for reading
    Set ih10File = fso.OpenTextFile(ih10FilePath, 1)
    Set zstkFile = fso.OpenTextFile(zstkFilePath, 1)
    
    ' Skip header lines in both files to get to actual data
    ' SAP exports typically have header information in the first two lines
    ih10File.SkipLine
    ih10File.SkipLine
    zstkFile.SkipLine
    zstkFile.SkipLine
    
    LogMessage "Reading IH10 data..."
    ' Process IH10 data (Stock Overview) - this contains the primary equipment information
    ' Each line represents a tool with its identifying information and location details
    Dim ih10Count
    ih10Count = 0
    
    Do Until ih10File.AtEndOfStream
        ih10Line = ih10File.ReadLine
        
        ' Skip empty lines that may appear in the export
        If Trim(ih10Line) <> "" Then
            ' Split tab-delimited line into individual field values
            ih10Fields = Split(ih10Line, vbTab)
            
            ' Output debug information for the first few rows to help with troubleshooting
            ' This helps identify field structure changes in SAP exports
            If ih10Count < 3 Then ' Log details for first 3 rows only
                LogMessage "IH10 Row " & ih10Count & " field count: " & UBound(ih10Fields) + 1
                For debugIdx = 0 To UBound(ih10Fields)
                    LogMessage "  Field[" & debugIdx & "]: " & ih10Fields(debugIdx)
                Next
            End If
            
            ' Ensure we have enough fields for processing (at least 7 fields expected)
            ' Field structure: [0]=empty, [1]=Material, [2]=SerialNumber, [3]=Equipment, [4]=ChangedOn, [5]=Plant, [6]=SLoc
            If UBound(ih10Fields) >= 6 Then
                Dim material, serialNum, equipment, plantVal, slocVal, changedDate, key
                
                ' Extract key information from each field
                material = Trim(ih10Fields(1))      ' Material number (12NC)
                serialNum = Trim(ih10Fields(2))     ' Tool serial number
                equipment = Trim(ih10Fields(3))     ' SAP equipment number
                plantVal = Trim(ih10Fields(5))      ' Plant code
                slocVal = Trim(ih10Fields(6))       ' Storage location code
                
                ' Create unique composite key for record identification and deduplication
                ' This key allows us to match records between IH10 and ZSTK datasets
                key = material & "|" & plantVal & "|" & slocVal & "|" & serialNum
                
                ' Create dictionary object to store this record's data
                Dim ih10Record
                Set ih10Record = CreateObject("Scripting.Dictionary")
                
                ' Store all relevant fields in the record dictionary
                ih10Record.Add "Material", material
                ih10Record.Add "SerialNumber", serialNum
                ih10Record.Add "Equipment", equipment
                ih10Record.Add "Plant", plantVal
                ih10Record.Add "SLOC", slocVal
                ih10Record.Add "BinLocation", ""  ' Will be populated from ZSTK data
                
                ' Convert date format from SAP format (DD.MM.YYYY) to standardized format (YYYYMMDD)
                ' This ensures consistent date formatting for downstream processing
                Dim dateStr, dateParts
                dateStr = Trim(ih10Fields(4)) ' ChangedOn column
                
                If dateStr <> "" And InStr(dateStr, ".") > 0 Then
                    dateParts = Split(dateStr, ".")
                    If UBound(dateParts) = 2 Then
                        ' Rearrange DD.MM.YYYY to YYYYMMDD format
                        dateStr = dateParts(2) & dateParts(1) & dateParts(0)
                    End If
                End If
                
                ih10Record.Add "ChangedDate", dateStr
                
                ' Add record to collection if it doesn't already exist (deduplication)
                If Not ih10Records.Exists(key) Then
                    ih10Records.Add key, ih10Record
                    ih10Count = ih10Count + 1
                End If
            End If
        End If
    Loop
    
    LogMessage "Read " & ih10Count & " records from IH10"
    
    LogMessage "Reading ZSTK data..."
    ' Process ZSTK data (Custom Stock Report) - this provides bin locations and profile information
    ' This data complements the IH10 data with additional location details
    Dim zstkCount
    zstkCount = 0
    
    Do Until zstkFile.AtEndOfStream
        zstkLine = zstkFile.ReadLine
        
        ' Skip empty lines that may appear in the export
        If Trim(zstkLine) <> "" Then
            ' Split tab-delimited line into individual field values
            zstkFields = Split(zstkLine, vbTab)
            
            ' Output debug information for the first few rows to help with troubleshooting
            ' This helps identify field structure changes in SAP exports
            If zstkCount < 3 Then ' Log details for first 3 rows only
                LogMessage "ZSTK Row " & zstkCount & " field count: " & UBound(zstkFields) + 1
                For debugIdx = 0 To UBound(zstkFields)
                    LogMessage "  Field[" & debugIdx & "]: " & zstkFields(debugIdx)
                Next
            End If
            
            ' Ensure we have enough fields for processing (at least 7 fields expected)
            ' Field structure: [0]=empty, [1]=Material, [2]=MaterialDesc, [3]=Plant, [4]=SLoc, [5]=Bin, [6]=Profl., [7]=Unrestr.
            If UBound(zstkFields) >= 6 Then
                Dim zstkMaterial, zstkPlant, zstkSloc, zstkBin, zstkProfl, zstkKey
                
                ' Extract key information from ZSTK fields
                zstkMaterial = Trim(zstkFields(1))  ' Material number
                zstkPlant = Trim(zstkFields(3))     ' Plant code
                zstkSloc = Trim(zstkFields(4))      ' Storage location code
                zstkBin = Trim(zstkFields(5))       ' Bin location (specific position within storage location)
                zstkProfl = ""                      ' Profile field (additional classification)
                If UBound(zstkFields) >= 6 Then
                    zstkProfl = Trim(zstkFields(6))
                End If
                
                ' Create composite key for matching with IH10 data
                ' This key structure allows us to link ZSTK bin/profile data with IH10 equipment data
                zstkKey = zstkMaterial & "|" & zstkPlant & "|" & zstkSloc
                
                ' Store bin location and profile information for cross-referencing with IH10 data
                If Not zstkRecords.Exists(zstkKey) Then
                    ' Create dictionary to store ZSTK-specific data
                    Set zstkData = CreateObject("Scripting.Dictionary")
                    zstkData.Add "Bin", zstkBin
                    zstkData.Add "Profl", zstkProfl
                    zstkRecords.Add zstkKey, zstkData
                    zstkCount = zstkCount + 1
                End If
            End If
        End If
    Loop
    
    LogMessage "Read " & zstkCount & " records from ZSTK"
    
    ' Close both data files after processing
    ih10File.Close
    zstkFile.Close
    
    ' Create header row for the final merged output file
    ' This provides clear column identification for downstream processing
    outputFile.WriteLine "Material|Serial Number|Equipment Number|Plant|SLOC|Profl.|Bin Location|Changed Date"
    
    LogMessage "Merging data and writing output..."
    ' Process and merge the two datasets to create comprehensive tool backlog information
    ' This combines equipment details from IH10 with location specifics from ZSTK
    Dim outputCount
    outputCount = 0
    
    ' Iterate through all IH10 records and enrich them with ZSTK data where available
    For Each key In ih10Records.Keys
        Dim rec, materialNum, serialNo, equipNum, plantCode, slocCode, binLoc, proflVal, changDate
        Dim parts, materialKey
        
        ' Extract all data from the current IH10 record
        Set rec = ih10Records(key)
        
        materialNum = rec("Material")     ' Tool material number
        serialNo = rec("SerialNumber")    ' Unique serial number
        equipNum = rec("Equipment")       ' SAP equipment number
        plantCode = rec("Plant")          ' Plant location
        slocCode = rec("SLOC")           ' Storage location
        changDate = rec("ChangedDate")    ' Last change date
        
        ' Create lookup key to find corresponding ZSTK data
        ' This key matches on material, plant, and storage location
        materialKey = materialNum & "|" & plantCode & "|" & slocCode
        
        ' Search for matching bin location and profile data in ZSTK records
        binLoc = ""
        proflVal = ""
        If zstkRecords.Exists(materialKey) Then
            Set zstkData = zstkRecords(materialKey)
            binLoc = zstkData("Bin")      ' Specific bin location within storage location
            proflVal = zstkData("Profl")  ' Profile/classification information
        End If
        
        ' Write complete merged record to output file
        ' Format: Material|SerialNumber|Equipment|Plant|SLOC|Profile|BinLocation|ChangeDate
        outputFile.WriteLine materialNum & "|" & _
                             serialNo & "|" & _
                             equipNum & "|" & _
                             plantCode & "|" & _
                             slocCode & "|" & _
                             proflVal & "|" & _
                             binLoc & "|" & _
                             changDate
        
        outputCount = outputCount + 1
    Next
    
    LogMessage "Wrote " & outputCount & " records to output file"
    dataCollected = (outputCount > 0)
Else
    ' Handle case where one or both export files are missing
    LogMessage "One or both export files are missing"
    If Not fso.FileExists(ih10FilePath) Then
        LogMessage "Missing: " & ih10FilePath
    End If
    If Not fso.FileExists(zstkFilePath) Then
        LogMessage "Missing: " & zstkFilePath
    End If
End If

' Close the output file to ensure all data is written
outputFile.Close

' Generate final results based on data collection success
If dataCollected Then
    LogMessage "SUCCESS: Data collection completed successfully"
    WScript.Echo "SUCCESS:" & dataOutputPath
Else
    LogMessage "WARNING: No data was collected"
    WScript.Echo "WARNING: No data was collected"
End If

' Cleanup function to properly close log file and release resources
' This ensures log file integrity and prevents resource leaks
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Write completion timestamp and separator to log
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        ' Close file handle and release object reference
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

' Execute final cleanup before script termination
CloseLogFile()